package drm.genetics.simple;

import drm.genetics.*;

import java.util.Vector;
import java.util.Iterator;

/**
 * <p>Title: Genetinių Algoritmų paketas</p>
 * <p>Description: Genetinio Algoritmo Modeliuotojas</p>
 *
 * SymetricCMRunner specifika:
 *   Vieno žingsnio metu jis sukuria naują individų kartą su tokiu pačiu
 * individų skaičiu.
 *   1. Pradžioje į ją įtraukia nurodytą procentą geriausių individų jų
 * nemodifikuojant, taip pat juos paliekant senoje kartoje. Geriausiųjų
 * procentas pateikiamas kaip tikimybė nuo 0 iki 1 ir perskaičiavus apvalinamas
 * į viršų.
 *   2. Trūkstami individai gaunami iš senosios kartos individų, juos kryžminant
 * arba mutuojant.
 *   Kryžminimo ir mutacijos tikimybė čia susieti dydžiai, nes į naują kartą
 * patenka individas tik po kryžminimo arba mutacijos. Kitaip sakant kryžminimo
 * ir mutacijos tikimybių suma yra lygi 1.
 *
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: KTU</p>
 * @author Mantas Balnys
 * @version 1.0
 */

public class SymetricCMRunner extends AbstractRunner {
  public SymetricCMRunner() {
  }

  /**
   * mutacijos tikimybė
   * šiuo atveju individas yra arba kryžminamas arba mutuoja, tai reiškia kad
   * mutacijos ir kryžminimo tikimybių suma lygi 1.
   */
  protected double mutationProbability = 0.1;
  /**
   * kryžminimo tikimybė
   * šiuo atveju individas yra arba kryžminamas arba mutuoja, tai reiškia kad
   * mutacijos ir kryžminimo tikimybių suma lygi 1.
   */
  protected double crossoverProbability = 0.9;

  public void setMutationProbability(double mutationProbability) {
    this.mutationProbability = mutationProbability;
    crossoverProbability = 1 - mutationProbability;
  }

  public void setCrossoverProbability(double crossoverProbability) {
    setMutationProbability(1 - crossoverProbability);
  }

  public double getCrossoverProbability() {
    return crossoverProbability;
  }

  public double getMutationProbability() {
    return mutationProbability;
  }

  /**
   * Individų procentas, kurie patenką į naują kartą be pasikeitimų.
   */
  protected double bestPassPercentage = 0.1;

  public void setBestPassPercentage(double bestPassPercentage) {
    this.bestPassPercentage = bestPassPercentage;
  }

  public double getBestPassPercentage() {
    return bestPassPercentage;
  }

  public void step() {
    try {
      int size = individuals.size();
      Vector newRase = new Vector(size);
      // copying best genomes
      int howMuchBest = (int)Math.ceil(bestPassPercentage * (double)size);
      Iterator it = individuals.iterator();
      while ((it.hasNext())&&(newRase.size() < howMuchBest)) {
        Individual ind = (Individual)it.next();
        Individual ind2 = (Individual)ind.getClass().newInstance();
        ind2.accessGenome(ind.getGenome());
        newRase.add(ind2);
      }
      // creating the rest
      while (newRase.size() < size) {
        if ((individuals.size() <= 1)||
            (random.nextDouble() < mutationProbability)) { // atliekama mutacija
          Individual ind = getRandomIndividual();
          task.getMutation().mutateSame(ind);
          newRase.add(ind);
        } else { // atliekamas kryžminimas
          Individual ind1 = getRandomIndividual();
          Individual ind2 = getRandomIndividual();
          task.getCrossover().crossSame(ind1, ind2);
          newRase.add(ind1);
          if (newRase.size() < size) newRase.add(ind2);
        }
      }
      // ending task
      individuals.clear();
      individuals.addAll(newRase);
      stepCount++;
    }
    catch (Exception ex) {
      ex.printStackTrace();
    }
  }

}